/**
 * \file sdc_intern.c
 *
 * \brief Common functions which are used internally only
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <arpa/inet.h>
#include <private/sdc_arch.h>
#include <private/sdc_intern.h>

/* Definitions types and defaults */

/* Functions */

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_check_init_dgst_tag_iv_output_buffer (uint8_t **data,
                                                             size_t *len,
                                                             size_t *internal_len,
                                                             size_t default_val,
                                                             bool *external_input)
{
    sdc_error_t err = SDC_OK;
    bool is_default_len = false;
    bool is_external_provided_buffer = false;


    if (!data) {
        err = SDC_INVALID_PARAMETER;
    } else {
        if (*data) {
            is_external_provided_buffer = true;
        }
    }

    if (!len) {
        err = SDC_INVALID_PARAMETER;
    } else {
        *internal_len = *len;

        if (*len == default_val) {
            is_default_len = true;
        }
    }

    if ((is_external_provided_buffer) &&
        ((!external_input) || (is_default_len))) {
        err = SDC_INVALID_PARAMETER;
        is_external_provided_buffer = false;
    } else {
        if (external_input) {
            *external_input = is_external_provided_buffer;
        }
    }

    if (!is_external_provided_buffer) {
        if (data) {
            *data = NULL;
        }
        if (len) {
            *len = 0;
        }
    }

    return err;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_check_init_data_output_buffer (uint8_t **data, size_t *len)
{
    sdc_error_t err = SDC_OK;

    if (!data) {
        err = SDC_INVALID_PARAMETER;
    } else {
        if (*data) {
            err = SDC_INVALID_PARAMETER;
            *data = NULL;
        }
    }

    if (!len) {
        err = SDC_INVALID_PARAMETER;
    } else {
        *len = 0;
    }

    return err;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_add_pkcs7_padding(uint8_t *buffer,
                                         size_t offset,
                                         uint8_t padding_bytes)
{
    uint8_t *ptr;

    if (padding_bytes == 0) {
        /* PKCS7 padding will be one up to 255 bytes */
        return SDC_PADDING_ERROR;
    }

    ptr = buffer + offset;

    memset(ptr, padding_bytes, padding_bytes);

    return SDC_OK;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_remove_pkcs7_padding(uint8_t *buffer,
                                            size_t *data_len, size_t block_alignment)
{
    uint8_t *ptr;
    uint8_t padding_bytes;
    uint8_t i;

    if (*data_len == 0) {
        /* PKCS7 padding will be one up to 255 bytes */
        return SDC_PADDING_ERROR;
    }

    ptr = buffer + *data_len - 1;
    padding_bytes = *ptr;

    if ((padding_bytes == 0) || (padding_bytes > block_alignment)) {
        /* PKCS7 padding will be one up to block_alignment bytes */
        return SDC_PADDING_ERROR;
    }

    if (padding_bytes > *data_len) {
        return SDC_PADDING_ERROR;
    }

    *data_len -= padding_bytes;

    for (i=0; i<(padding_bytes-1); i++) {
        ptr--;
        if (*ptr != padding_bytes) {
            /* all padded bytes need to be equal to the number of padded bytes*/
            return SDC_PADDING_ERROR;
        }
    }

    return SDC_OK;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_check_data_input_buffer (const uint8_t *data, size_t len)
{
    sdc_error_t err = SDC_OK;

    if ((data == NULL) && !(len == 0))
        err = SDC_INVALID_PARAMETER;

    return err;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_check_tag_iv_input_buffer (const uint8_t *data, size_t len,
                                                  size_t default_val)
{
    sdc_error_t err = SDC_OK;

    if (len == default_val)
        err = SDC_INVALID_PARAMETER;

    if ((data == NULL) && !(len == 0))
        err = SDC_INVALID_PARAMETER;

    return err;
}

static sdc_error_t sdc_intern_plain_cipher_spec_check_min_max_align(
    size_t in_len,
    const sdc_plain_cipher_spec_t *plain_cipher_spec,
    const size_t block_len)
{
    if (in_len < plain_cipher_spec->min)
        return SDC_IN_DATA_INVALID;

    if (in_len > plain_cipher_spec->max)
        return SDC_IN_DATA_INVALID;

    if ((plain_cipher_spec->aligned) &&
        (0 != (in_len % block_len)))
        return SDC_IN_DATA_INVALID;

    return SDC_OK;
}

sdc_error_t sdc_intern_inout_input_check_min_max_align(
    size_t in_len,
    const sdc_inout_spec_t *spec,
    bool plain_input)
{
    if (plain_input)
        return sdc_intern_plain_cipher_spec_check_min_max_align(
                in_len,
                &(spec->plain),
                spec->block_len);

    return sdc_intern_plain_cipher_spec_check_min_max_align(
                in_len,
                &(spec->cipher),
                spec->block_len);
}

sdc_error_t sdc_intern_input_only_check_min_max_align(
    size_t in_len,
    const sdc_in_spec_t *spec)
{
    return sdc_intern_plain_cipher_spec_check_min_max_align(
                in_len,
                &(spec->plain),
                spec->block_len);
}

static sdc_error_t sdc_intern_plain_cipher_spec_update_wont_exceed_max(
    size_t prev_in_len,
    size_t curr_in_len,
    const sdc_plain_cipher_spec_t *plain_cipher_spec)
{
    size_t remaining_in_len;

    /*
     * The functions should only process the data if the max wouldn't be
     * exceeded - so this should not happen.
     * The additional check is done to prevent overflows due to implementation errors.
     */
    if (prev_in_len > plain_cipher_spec->max)
        return SDC_INTERNAL_ERROR;

    remaining_in_len = plain_cipher_spec->max - prev_in_len;

    if (curr_in_len > remaining_in_len)
        return SDC_IN_DATA_INVALID;

    return SDC_OK;
}

sdc_error_t sdc_intern_inout_input_check_update_wont_exceed_max(
    size_t prev_in_len,
    size_t curr_in_len,
    const sdc_inout_spec_t *spec,
    bool plain_input)
{
    if (plain_input)
        return sdc_intern_plain_cipher_spec_update_wont_exceed_max(
                    prev_in_len,
                    curr_in_len,
                    &(spec->plain));

    return sdc_intern_plain_cipher_spec_update_wont_exceed_max(
                prev_in_len,
                curr_in_len,
                &(spec->cipher));
}

sdc_error_t sdc_intern_input_only_check_update_wont_exceed_max(
    size_t prev_in_len,
    size_t curr_in_len,
    const sdc_in_spec_t *spec)
{
    return sdc_intern_plain_cipher_spec_update_wont_exceed_max(
                prev_in_len,
                curr_in_len,
                &(spec->plain));
}

sdc_error_t sdc_intern_range_min_max_mod_check(
    size_t len,
    const sdc_range_t *val)
{
    if (len < val->min)
        return SDC_INVALID_PARAMETER;

    if (len > val->max)
        return SDC_INVALID_PARAMETER;

    if ((val->mod > 1) && (0 != (len % val->mod)))
        return SDC_INVALID_PARAMETER;

    return SDC_OK;
}

sdc_error_t sdc_intern_range_get_min_max_mod_dflt(
    const sdc_range_t *val,
    size_t *min_val,
    size_t *max_val,
    size_t *mod,
    size_t *default_val)
{
    if (min_val)
        *min_val = val->min;

    if (max_val)
        *max_val = val->max;

    if (mod)
        *mod = val->mod;

    if (default_val)
        *default_val = val->dflt;

    return SDC_OK;
}

/* defined in sdc_intern.h */
void sdc_intern_session_key_info_init_overwrite(sdc_key_info_t *key_info)
{
    key_info->key_type = SDC_KEY_TYPE_INVALID;
    key_info->key_fmt = SDC_KEY_FMT_UNKNOWN;
    key_info->key_len_bits = 0;
    key_info->key_len_bytes = 0;
    key_info->kid = SDC_FLAG_INVALID_KID;
    key_info->perms.uid = 0;
    key_info->perms.gid = 0;
    key_info->perms.perms_owner = SDC_PERM_NONE;
    key_info->perms.perms_group = SDC_PERM_NONE;
    key_info->perms.perms_others = SDC_PERM_NONE;
    key_info->perms.perms_inherit_owner = SDC_PERM_NONE;
    key_info->perms.perms_inherit_group = SDC_PERM_NONE;
    key_info->perms.perms_inherit_others = SDC_PERM_NONE;
    key_info->key_modifier.data = NULL;
    key_info->key_modifier.len = 0;
    key_info->key_modifier.is_secret = false;
    key_info->authenticity = SDC_KEY_AUTH_UNKNOWN;
    key_info->confidentiality = SDC_KEY_CONFIDENT_UNKNOWN;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_session_key_info_get(sdc_session_t *session,
                                            sdc_key_info_t *key_info, bool with_modifier)
{
    sdc_error_t err;

    free(key_info->key_modifier.data);

    sdc_intern_session_key_info_init_overwrite(key_info);

    err = sdc_arch_session_key_info_get(session, key_info, with_modifier);

    if (err == SDC_OK) {
        if (key_info->key_type == SDC_KEY_TYPE_INVALID)
            err = SDC_KEY_UNSET;

        /* make sure secret modifiers are never exported */
        if ((err == SDC_OK) && (key_info->key_modifier.is_secret) &&
            ((key_info->key_modifier.data) || (key_info->key_modifier.len > 0)))
            err = SDC_INTERNAL_ERROR;

        if (err != SDC_OK)
            free(key_info->key_modifier.data);
    }

    return err;
}

void sdc_inter_form_header_init(sdc_form_header_generic_t *header)
{
    memset(header, 0, sizeof(sdc_form_header_generic_t));
}

void sdc_inter_form_header_free(sdc_form_header_generic_t *header)
{
    if (header->key_info.common.type == SDC_FORMATTED_KEY_BUILT_IN)
        free(header->key_info.builtin.key_modifier.data);
}

static sdc_error_t sdc_intern_fill_key_in_generic_header(sdc_session_t *session,
                                                         sdc_form_header_generic_t *header)
{
    sdc_error_t err = SDC_OK;
    sdc_key_info_t key_info;

    sdc_intern_session_key_info_init_overwrite(&key_info);

    err = sdc_intern_session_key_info_get(session, &key_info, true);
    if (err != SDC_OK)
        return err;

    header->header_len += sizeof(sdc_form_header_key_info_common_v0_t);

    switch (key_info.key_type) {
    case SDC_KEY_TYPE_INVALID:
        err = SDC_KEY_INFO_INVALID; /* this should never happen */
        break;
    case SDC_KEY_TYPE_KEYSTORE_PLAIN:
    /* FALLTHROUGH */
    case SDC_KEY_TYPE_KEYSTORE_RANDOM:
    /* FALLTHROUGH */
    case SDC_KEY_TYPE_KEYSTORE_IMPORTED:
    /* FALLTHROUGH */
    case SDC_KEY_TYPE_KEYSTORE_PRODUCT:
        header->key_info.common.type = SDC_FORMATTED_KEY_STORAGE;

        header->key_info.keystore.kid = key_info.kid;

        free(key_info.key_modifier.data); /* we don't need the modifier */

        header->header_len += sizeof(sdc_form_header_key_info_keystore_v0_t);

        break;
    case SDC_KEY_TYPE_BUILT_IN:
        header->key_info.common.type = SDC_FORMATTED_KEY_BUILT_IN;

        header->key_info.builtin.key_fmt = key_info.key_fmt;
        header->key_info.builtin.key_len_bits = key_info.key_len_bits;
        header->key_info.builtin.key_len_bytes = key_info.key_len_bytes;

        memcpy(&header->key_info.builtin.perms,
               &key_info.perms, sizeof(sdc_permissions_t));

        /* modifier will be freed using sdc_inter_form_header_free */
        header->key_info.builtin.key_modifier = key_info.key_modifier;

        /* sdc_intern_session_key_info_get will ensure that
         * secret_modifier is false if modifier_len > 0 */

        header->header_len += sizeof(sdc_form_header_key_info_builtin_v0_t);
        header->header_len += header->key_info.builtin.key_modifier.len;
        break;
    default:
        err = SDC_INTERNAL_ERROR; /* key type not supported */
        break;
    }

    return err;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_wrap_unwrap_formatted_fill_header (sdc_session_t *session,
                                                          const sdc_wrap_unwrap_type_t *type,
                                                          size_t data_len,
                                                          size_t iv_len,
                                                          size_t tag_len,
                                                          size_t aad_len,
                                                          sdc_form_header_generic_t *header)
{

    sdc_error_t err;

    err = sdc_wrap_unwrap_type_get_algorithm(type, &(header->wrap_unwrap.alg));

    if (err == SDC_OK)
        err = sdc_wrap_unwrap_type_get_block_mode(type, &(header->wrap_unwrap.blk));

    if (err == SDC_OK) {
        uint64_t opt_bmsk;

        err = sdc_wrap_unwrap_type_get_opt_bmsk(type, &opt_bmsk);

        /* formatted header doesn't support additional options */
        if ((err == SDC_OK) && (opt_bmsk != 0))
            err = SDC_INTERNAL_ERROR;
    }

    if (err == SDC_OK) {

        header->header_len = sizeof(sdc_form_header_ext_t);
        header->overall_formatted_len = 0;

        err = sdc_intern_fill_key_in_generic_header(session, header);

        header->operation = SDC_FORMATTED_TYPE_WRAP_UNWRAP;
        header->wrap_unwrap.data_len = data_len;
        header->overall_formatted_len += data_len;
        header->wrap_unwrap.iv_len = iv_len;
        header->overall_formatted_len += iv_len;
        header->wrap_unwrap.tag_len = tag_len;
        header->overall_formatted_len += tag_len;
        header->wrap_unwrap.aad_len = aad_len;
        header->overall_formatted_len += aad_len;
        header->header_len += sizeof(sdc_form_header_wrap_unwrap_v0_t);

        header->overall_formatted_len += header->header_len;
    }

    return err;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_wrap_unwrap_formatted_update_pointers (
    sdc_form_header_generic_t *header,
    uint8_t *buf_ptr)
{

    header->wrap_unwrap.data = buf_ptr + header->header_len;
    header->wrap_unwrap.iv   = header->wrap_unwrap.data + header->wrap_unwrap.data_len;
    header->wrap_unwrap.tag  = header->wrap_unwrap.iv + header->wrap_unwrap.iv_len;
    header->wrap_unwrap.aad  = header->wrap_unwrap.tag + header->wrap_unwrap.tag_len;

    return SDC_OK;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_encrypt_decrypt_formatted_fill_header (sdc_session_t *session,
                                                              const sdc_encrypt_decrypt_type_t *type,
                                                              size_t data_len,
                                                              size_t iv_len,
                                                              sdc_form_header_generic_t *header)
{

    sdc_error_t err;

    err = sdc_encrypt_decrypt_type_get_algorithm(type, &(header->encrypt_decrypt.alg));

    if (err == SDC_OK)
        err = sdc_encrypt_decrypt_type_get_block_mode(type, &(header->encrypt_decrypt.blk));

    if (err == SDC_OK) {
        uint64_t opt_bmsk;

        err = sdc_encrypt_decrypt_type_get_opt_bmsk(type, &opt_bmsk);

        /* formatted header doesn't support additional options */
        if ((err == SDC_OK) && (opt_bmsk != 0))
            err = SDC_INTERNAL_ERROR;
    }

    if (err == SDC_OK) {

        header->header_len = sizeof(sdc_form_header_ext_t);
        header->overall_formatted_len = 0;

        err = sdc_intern_fill_key_in_generic_header(session, header);

        header->operation = SDC_FORMATTED_TYPE_ENCRYPT_DECRYPT;
        header->encrypt_decrypt.data_len = data_len;
        header->overall_formatted_len += data_len;
        header->encrypt_decrypt.iv_len = iv_len;
        header->overall_formatted_len += iv_len;
        header->header_len += sizeof(sdc_form_header_encrypt_decrypt_v0_t);

        header->overall_formatted_len += header->header_len;
    }

    return err;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_encrypt_decrypt_formatted_update_pointers (
    sdc_form_header_generic_t *header,
    uint8_t *buf_ptr)
{

    header->encrypt_decrypt.data = buf_ptr + header->header_len;
    header->encrypt_decrypt.iv   = header->encrypt_decrypt.data + header->encrypt_decrypt.data_len;

    return SDC_OK;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_sign_verify_formatted_fill_header (sdc_session_t *session,
                                                          const sdc_sign_verify_type_t *type,
                                                          size_t data_len,
                                                          size_t iv_len,
                                                          size_t tag_len,
                                                          sdc_form_header_generic_t *header)
{

    sdc_error_t err;

    err = sdc_sign_verify_type_get_alg(type, &(header->sign_verify.alg));

    if (err == SDC_OK)
        err = sdc_sign_verify_type_get_hash(type, &(header->sign_verify.hash));

    if (err == SDC_OK) {
        uint64_t opt_bmsk;

        err = sdc_sign_verify_type_get_opt_bmsk(type, &opt_bmsk);

        /* formatted header doesn't support additional options */
        if ((err == SDC_OK) && (opt_bmsk != 0))
            err = SDC_INTERNAL_ERROR;
    }

    if (err == SDC_OK) {

        header->header_len = sizeof(sdc_form_header_ext_t);
        header->overall_formatted_len = 0;

        err = sdc_intern_fill_key_in_generic_header(session, header);

        header->operation = SDC_FORMATTED_TYPE_SIGN_VERIFY;
        header->sign_verify.data_len = data_len;
        header->overall_formatted_len += data_len;
        header->sign_verify.iv_len = iv_len;
        header->overall_formatted_len += iv_len;
        header->sign_verify.tag_len = tag_len;
        header->overall_formatted_len += tag_len;
        header->header_len += sizeof(sdc_form_header_sign_verify_v0_t);

        header->overall_formatted_len += header->header_len;
    }

    return err;
}

/* defined in sdc_intern.h */
sdc_error_t sdc_intern_sign_verify_formatted_update_pointers (
    sdc_form_header_generic_t *header,
    uint8_t *buf_ptr)
{

    header->sign_verify.data = buf_ptr + header->header_len;
    header->sign_verify.iv   = header->sign_verify.data + header->sign_verify.data_len;
    header->sign_verify.tag  = header->sign_verify.iv + header->sign_verify.iv_len;

    return SDC_OK;
}

static void sdc_intern_formatted_sdc_perm_to_form_perm(sdc_error_t *err,
                                                       const sdc_perm_bmsk_t in_perm, uint32_t *out_perm)
{
    if ((in_perm & 0xFFFF) != in_perm)
        *err = SDC_INTERNAL_ERROR;
    else
        *out_perm = htonl((uint32_t)in_perm);
}

static void sdc_intern_formatted_form_perm_to_sdc_perm(sdc_error_t *err,
                                                       const uint32_t in_perm, sdc_perm_bmsk_t *out_perm)
{
    uint32_t tmp = ntohl(in_perm);

    if ((tmp & 0xFFFF) != tmp)
        *err = SDC_FORMATTED_DATA_INVALID;
    else
        *out_perm = (sdc_perm_bmsk_t)tmp;
}

static void sdc_intern_formatted_sdc_bool_to_form_bool(sdc_error_t *err,
                                                       const bool in_bool, uint8_t *out_bool)
{
    (void)err;
    if (in_bool)
        *out_bool = 0xFF;
    else
        *out_bool = 0x00;
}

static void sdc_intern_formatted_form_bool_to_sdc_bool(sdc_error_t *err,
                                                       const uint8_t in_bool, bool *out_bool)
{
    if (in_bool == 0x00)
        *out_bool = false;
    else if (in_bool == 0xFF)
        *out_bool = true;
    else
        *err = SDC_FORMATTED_DATA_INVALID;
}

static void sdc_intern_formatted_sdc_uid_to_form_uid(sdc_error_t *err,
                                                     const uid_t in_uid, uint32_t *out_uid)
{
    (void)err;
    if (in_uid == SDC_FLAG_INVALID_UID)
        *out_uid = SDC_INTERN_FORMATTED_INVAL_UID;
    else
        *out_uid = htonl((uint32_t)in_uid);
}

static void sdc_intern_formatted_form_uid_to_sdc_uid(sdc_error_t *err,
                                                     const uint32_t in_uid, uid_t *out_uid)
{
    (void)err;
    if (in_uid == SDC_INTERN_FORMATTED_INVAL_UID)
        *out_uid = SDC_FLAG_INVALID_UID;
    else
        *out_uid = (uid_t)ntohl(in_uid);
}

static void sdc_intern_formatted_sdc_gid_to_form_gid(sdc_error_t *err,
                                                     const gid_t in_gid, uint32_t *out_gid)
{
    (void)err;
    if (in_gid == SDC_FLAG_INVALID_GID)
        *out_gid = SDC_INTERN_FORMATTED_INVAL_GID;
    else
        *out_gid = htonl((uint32_t)in_gid);
}

static void sdc_intern_formatted_form_gid_to_sdc_gid(sdc_error_t *err,
                                                     const uint32_t in_gid, uid_t *out_gid)
{
    (void)err;
    if (in_gid == SDC_INTERN_FORMATTED_INVAL_GID)
        *out_gid = SDC_FLAG_INVALID_GID;
    else
        *out_gid = (gid_t)ntohl(in_gid);
}


/* defined in sdc_intern.h */
sdc_error_t sdc_intern_formatted_write_header (
    const sdc_form_header_generic_t *gen_header,
    void *buf_ptr)
{

    sdc_error_t err = SDC_OK;
    sdc_form_header_ext_t *main_header;
    sdc_form_header_key_info_common_v0_t *key_common_header;
    sdc_form_header_key_info_keystore_v0_t *key_keystore_header;
    sdc_form_header_key_info_builtin_v0_t *key_builtin_header;
    size_t key_info_len;
    sdc_form_header_wrap_unwrap_v0_t *wrap_unwrap_header;
    sdc_form_header_encrypt_decrypt_v0_t *encrypt_decrypt_header;
    sdc_form_header_sign_verify_v0_t *sign_verify_header;
    void *next_ptr;

    next_ptr = buf_ptr;

    main_header = (sdc_form_header_ext_t*)next_ptr;
    next_ptr = ((uint8_t*)next_ptr + sizeof(sdc_form_header_ext_t));
    main_header->magic = htonl((uint32_t)SDC_FORMATTED_MAGIC);


    key_common_header = (sdc_form_header_key_info_common_v0_t*)next_ptr;
    next_ptr = ((uint8_t*)next_ptr + sizeof(sdc_form_header_key_info_common_v0_t));
    key_info_len = sizeof(sdc_form_header_key_info_common_v0_t);
    main_header->key_info_version = 0;
    key_common_header->key_type = htonl((uint32_t)gen_header->key_info.common.type);

    /* fill key part */
    switch (gen_header->key_info.common.type) {
    case SDC_FORMATTED_KEY_STORAGE:
        key_keystore_header = (sdc_form_header_key_info_keystore_v0_t*)next_ptr;
        next_ptr = ((uint8_t*)next_ptr + sizeof(sdc_form_header_key_info_keystore_v0_t));
        key_info_len += sizeof(sdc_form_header_key_info_keystore_v0_t);
        key_keystore_header->kid = htonl((uint32_t)gen_header->key_info.keystore.kid);
        break;
    case SDC_FORMATTED_KEY_BUILT_IN:
        key_builtin_header = (sdc_form_header_key_info_builtin_v0_t*)next_ptr;
        next_ptr = ((uint8_t*)next_ptr + sizeof(sdc_form_header_key_info_builtin_v0_t));
        key_info_len += sizeof(sdc_form_header_key_info_builtin_v0_t);

        sdc_intern_formatted_sdc_uid_to_form_uid(&err, gen_header->key_info.builtin.perms.uid, &key_builtin_header->uid);
        sdc_intern_formatted_sdc_gid_to_form_gid(&err, gen_header->key_info.builtin.perms.gid, &key_builtin_header->gid);
        key_builtin_header->secret_type = htonl((uint32_t)gen_header->key_info.builtin.key_fmt);
        key_builtin_header->key_len_bits = htonl(gen_header->key_info.builtin.key_len_bits);
        key_builtin_header->key_len_bytes = htonl(gen_header->key_info.builtin.key_len_bytes);
        sdc_intern_formatted_sdc_perm_to_form_perm(&err, gen_header->key_info.builtin.perms.perms_owner, &key_builtin_header->perms_owner);
        sdc_intern_formatted_sdc_perm_to_form_perm(&err, gen_header->key_info.builtin.perms.perms_group, &key_builtin_header->perms_group);
        sdc_intern_formatted_sdc_perm_to_form_perm(&err, gen_header->key_info.builtin.perms.perms_others, &key_builtin_header->perms_others);
        sdc_intern_formatted_sdc_perm_to_form_perm(&err, gen_header->key_info.builtin.perms.perms_inherit_owner, &key_builtin_header->perms_inherit_owner);
        sdc_intern_formatted_sdc_perm_to_form_perm(&err, gen_header->key_info.builtin.perms.perms_inherit_group, &key_builtin_header->perms_inherit_group);
        sdc_intern_formatted_sdc_perm_to_form_perm(&err, gen_header->key_info.builtin.perms.perms_inherit_others, &key_builtin_header->perms_inherit_others);
        key_builtin_header->modifier_len = htonl(gen_header->key_info.builtin.key_modifier.len);
        sdc_intern_formatted_sdc_bool_to_form_bool(&err, gen_header->key_info.builtin.key_modifier.is_secret, &key_builtin_header->secret_modifier);

        if (gen_header->key_info.builtin.key_modifier.data) {
            memcpy(next_ptr,
                   gen_header->key_info.builtin.key_modifier.data,
                   gen_header->key_info.builtin.key_modifier.len);
            next_ptr = ((uint8_t*)next_ptr + gen_header->key_info.builtin.key_modifier.len);
            key_info_len += gen_header->key_info.builtin.key_modifier.len;
        }
        break;
    default:
        err = SDC_INTERNAL_ERROR;
    }

    main_header->key_info_len = htonl(key_info_len);

    if (err == SDC_OK) {
        main_header->format_type = htonl((uint32_t)gen_header->operation);

        switch (gen_header->operation) {
        case SDC_FORMATTED_TYPE_WRAP_UNWRAP:
            /* fill wrap part */
            main_header->format_len = htonl(sizeof(sdc_form_header_wrap_unwrap_v0_t));
            main_header->format_type_version = 0;
            wrap_unwrap_header = (sdc_form_header_wrap_unwrap_v0_t*)next_ptr;
            next_ptr = ((uint8_t*)next_ptr + sizeof(sdc_form_header_wrap_unwrap_v0_t));
            wrap_unwrap_header->alg = htonl((uint32_t)gen_header->wrap_unwrap.alg-(uint32_t)SDC_WRAP_ALG_FIRST);
            wrap_unwrap_header->blk = htonl((uint32_t)gen_header->wrap_unwrap.blk-(uint32_t)SDC_WRAP_BLK_FIRST);
            wrap_unwrap_header->data_len = htonl((uint32_t)gen_header->wrap_unwrap.data_len);
            wrap_unwrap_header->iv_len = htonl((uint32_t)gen_header->wrap_unwrap.iv_len);
            wrap_unwrap_header->tag_len = htonl((uint32_t)gen_header->wrap_unwrap.tag_len);
            wrap_unwrap_header->aad_len = htonl((uint32_t)gen_header->wrap_unwrap.aad_len);
            break;
        case SDC_FORMATTED_TYPE_ENCRYPT_DECRYPT:
            /* fill wrap part */
            main_header->format_len = htonl(sizeof(sdc_form_header_encrypt_decrypt_v0_t));
            main_header->format_type_version = 0;
            encrypt_decrypt_header = (sdc_form_header_encrypt_decrypt_v0_t*)next_ptr;
            next_ptr = ((uint8_t*)next_ptr + sizeof(sdc_form_header_encrypt_decrypt_v0_t));
            encrypt_decrypt_header->alg = htonl((uint32_t)gen_header->encrypt_decrypt.alg-(uint32_t)SDC_ENCDEC_ALG_FIRST);
            encrypt_decrypt_header->blk = htonl((uint32_t)gen_header->encrypt_decrypt.blk-(uint32_t)SDC_ENCDEC_BLK_FIRST);
            encrypt_decrypt_header->data_len = htonl((uint32_t)gen_header->encrypt_decrypt.data_len);
            encrypt_decrypt_header->iv_len = htonl((uint32_t)gen_header->encrypt_decrypt.iv_len);
            break;
        case SDC_FORMATTED_TYPE_SIGN_VERIFY:
            /* fill wrap part */
            main_header->format_len = htonl(sizeof(sdc_form_header_sign_verify_v0_t));
            main_header->format_type_version = 0;
            sign_verify_header = (sdc_form_header_sign_verify_v0_t*)next_ptr;
            next_ptr = ((uint8_t*)next_ptr + sizeof(sdc_form_header_sign_verify_v0_t));
            sign_verify_header->alg = htonl((uint32_t)gen_header->sign_verify.alg-(uint32_t)SDC_SIGNVER_ALG_FIRST);
            sign_verify_header->hash = htonl((uint32_t)gen_header->sign_verify.hash-(uint32_t)SDC_SIGNVER_HASH_FIRST);
            sign_verify_header->data_len = htonl((uint32_t)gen_header->sign_verify.data_len);
            sign_verify_header->iv_len = htonl((uint32_t)gen_header->sign_verify.iv_len);
            sign_verify_header->tag_len = htonl((uint32_t)gen_header->sign_verify.tag_len);
            break;
        default:
            err = SDC_INTERNAL_ERROR;
        }
    }

    return err;
}

static sdc_error_t sdc_intern_formatted_read_header_opswitch (
    sdc_form_header_generic_t *gen_header,
    const void *formatted_ptr,
    const sdc_form_header_ext_t *main_header,
    sdc_form_operation_t operation,
    size_t *remaining_ptr,
    size_t *offset_ptr )
{
    sdc_error_t err = SDC_OK;
    const sdc_form_header_wrap_unwrap_v0_t *wrap_unwrap_header;
    const sdc_form_header_encrypt_decrypt_v0_t *encrypt_decrypt_header;
    const sdc_form_header_sign_verify_v0_t *sign_verify_header;

    switch (operation) {
    case SDC_FORMATTED_TYPE_WRAP_UNWRAP:
        gen_header->operation = SDC_FORMATTED_TYPE_WRAP_UNWRAP;
        switch (ntohl(main_header->format_type_version)) {
        case 0:
            /* check if this is wrap/unwrap version 0 */
            if ((*remaining_ptr < sizeof(sdc_form_header_wrap_unwrap_v0_t)) ||
                (ntohl(main_header->format_len) != sizeof(sdc_form_header_wrap_unwrap_v0_t))) {
                err = SDC_FORMATTED_DATA_INVALID;
            } else {
                wrap_unwrap_header = (const sdc_form_header_wrap_unwrap_v0_t*)(void *)((uint8_t*)formatted_ptr + (*offset_ptr));;
                (*offset_ptr) += sizeof(sdc_form_header_wrap_unwrap_v0_t);
                (*remaining_ptr) -= sizeof(sdc_form_header_wrap_unwrap_v0_t);
                gen_header->header_len += sizeof(sdc_form_header_wrap_unwrap_v0_t);
                gen_header->overall_formatted_len += sizeof(sdc_form_header_wrap_unwrap_v0_t);

                gen_header->wrap_unwrap.alg = (sdc_wrap_unwrap_alg_t)(ntohl(wrap_unwrap_header->alg)+(uint32_t)SDC_WRAP_ALG_FIRST);
                gen_header->wrap_unwrap.blk = (sdc_wrap_unwrap_blk_t)(ntohl(wrap_unwrap_header->blk)+(uint32_t)SDC_WRAP_BLK_FIRST);
                gen_header->wrap_unwrap.data_len = (size_t)ntohl(wrap_unwrap_header->data_len);
                gen_header->wrap_unwrap.iv_len = (size_t)ntohl(wrap_unwrap_header->iv_len);
                gen_header->wrap_unwrap.tag_len = (size_t)ntohl(wrap_unwrap_header->tag_len);
                gen_header->wrap_unwrap.aad_len = (size_t)ntohl(wrap_unwrap_header->aad_len);
            }

            if (err == SDC_OK) {
                if (gen_header->wrap_unwrap.data_len > *remaining_ptr) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    (*remaining_ptr) -= gen_header->wrap_unwrap.data_len;
                    gen_header->overall_formatted_len += gen_header->wrap_unwrap.data_len;
                }
            }

            if (err == SDC_OK) {
                if (gen_header->wrap_unwrap.iv_len > *remaining_ptr) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    (*remaining_ptr) -= gen_header->wrap_unwrap.iv_len;
                    gen_header->overall_formatted_len += gen_header->wrap_unwrap.iv_len;
                }
            }

            if (err == SDC_OK) {
                if (gen_header->wrap_unwrap.tag_len > *remaining_ptr) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    (*remaining_ptr) -= gen_header->wrap_unwrap.tag_len;
                    gen_header->overall_formatted_len += gen_header->wrap_unwrap.tag_len;
                }
            }

            if (err == SDC_OK) {
                if (gen_header->wrap_unwrap.aad_len > *remaining_ptr) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    (*remaining_ptr) -= gen_header->wrap_unwrap.aad_len;
                    gen_header->overall_formatted_len += gen_header->wrap_unwrap.aad_len;
                }
            }
            break;
        default:
            err = SDC_FORMATTED_DATA_INVALID;
        }
        break;
    case SDC_FORMATTED_TYPE_ENCRYPT_DECRYPT:
        gen_header->operation = SDC_FORMATTED_TYPE_ENCRYPT_DECRYPT;
        switch (ntohl(main_header->format_type_version)) {
        case 0:
            /* check if this is encrypt/decrypt version 0 */
            if ((*remaining_ptr < sizeof(sdc_form_header_encrypt_decrypt_v0_t)) ||
                (ntohl(main_header->format_len) != sizeof(sdc_form_header_encrypt_decrypt_v0_t))) {
                err = SDC_FORMATTED_DATA_INVALID;
            } else {
                encrypt_decrypt_header = (const sdc_form_header_encrypt_decrypt_v0_t*)(void *)((uint8_t*)formatted_ptr + (*offset_ptr));;
                (*offset_ptr) += sizeof(sdc_form_header_encrypt_decrypt_v0_t);
                (*remaining_ptr) -= sizeof(sdc_form_header_encrypt_decrypt_v0_t);
                gen_header->header_len += sizeof(sdc_form_header_encrypt_decrypt_v0_t);
                gen_header->overall_formatted_len += sizeof(sdc_form_header_encrypt_decrypt_v0_t);

                gen_header->encrypt_decrypt.alg = (sdc_encrypt_decrypt_alg_t)(ntohl(encrypt_decrypt_header->alg)+(uint32_t)SDC_ENCDEC_ALG_FIRST);
                gen_header->encrypt_decrypt.blk = (sdc_encrypt_decrypt_blk_t)(ntohl(encrypt_decrypt_header->blk)+(uint32_t)SDC_ENCDEC_BLK_FIRST);
                gen_header->encrypt_decrypt.data_len = (size_t)ntohl(encrypt_decrypt_header->data_len);
                gen_header->encrypt_decrypt.iv_len = (size_t)ntohl(encrypt_decrypt_header->iv_len);
            }

            if (err == SDC_OK) {
                if (gen_header->encrypt_decrypt.data_len > *remaining_ptr) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    (*remaining_ptr) -= gen_header->encrypt_decrypt.data_len;
                    gen_header->overall_formatted_len += gen_header->encrypt_decrypt.data_len;
                }
            }

            if (err == SDC_OK) {
                if (gen_header->encrypt_decrypt.iv_len > *remaining_ptr) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    (*remaining_ptr) -= gen_header->encrypt_decrypt.iv_len;
                    gen_header->overall_formatted_len += gen_header->encrypt_decrypt.iv_len;
                }
            }
            break;
        default:
            err = SDC_FORMATTED_DATA_INVALID;
        }
        break;
    case SDC_FORMATTED_TYPE_SIGN_VERIFY:
        gen_header->operation = SDC_FORMATTED_TYPE_SIGN_VERIFY;
        switch (ntohl(main_header->format_type_version)) {
        case 0:
            /* check if this is sign/verify version 0 */
            if ((*remaining_ptr < sizeof(sdc_form_header_sign_verify_v0_t)) ||
                (ntohl(main_header->format_len) != sizeof(sdc_form_header_sign_verify_v0_t))) {
                err = SDC_FORMATTED_DATA_INVALID;
            } else {
                sign_verify_header = (const sdc_form_header_sign_verify_v0_t*)(void *)((uint8_t*)formatted_ptr + (*offset_ptr));
                (*offset_ptr) += sizeof(sdc_form_header_sign_verify_v0_t);
                (*remaining_ptr) -= sizeof(sdc_form_header_sign_verify_v0_t);
                gen_header->header_len += sizeof(sdc_form_header_sign_verify_v0_t);
                gen_header->overall_formatted_len += sizeof(sdc_form_header_sign_verify_v0_t);

                gen_header->sign_verify.alg = (sdc_sign_verify_alg_t)(ntohl(sign_verify_header->alg)+(uint32_t)SDC_SIGNVER_ALG_FIRST);
                gen_header->sign_verify.hash = (sdc_sign_verify_hash_t)(ntohl(sign_verify_header->hash)+(uint32_t)SDC_SIGNVER_HASH_FIRST);
                gen_header->sign_verify.data_len = (size_t)ntohl(sign_verify_header->data_len);
                gen_header->sign_verify.iv_len = (size_t)ntohl(sign_verify_header->iv_len);
                gen_header->sign_verify.tag_len = (size_t)ntohl(sign_verify_header->tag_len);
            }

            if (err == SDC_OK) {
                if (gen_header->sign_verify.data_len > *remaining_ptr) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    (*remaining_ptr) -= gen_header->sign_verify.data_len;
                    gen_header->overall_formatted_len += gen_header->sign_verify.data_len;
                }
            }

            if (err == SDC_OK) {
                if (gen_header->sign_verify.iv_len > *remaining_ptr) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    (*remaining_ptr) -= gen_header->sign_verify.iv_len;
                    gen_header->overall_formatted_len += gen_header->sign_verify.iv_len;
                }
            }

            if (err == SDC_OK) {
                if (gen_header->sign_verify.tag_len > *remaining_ptr) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    (*remaining_ptr) -= gen_header->sign_verify.tag_len;
                    gen_header->overall_formatted_len += gen_header->sign_verify.tag_len;
                }
            }
            break;
        default:
            err = SDC_FORMATTED_DATA_INVALID;
        }
        break;
    default:
        err = SDC_NOT_SUPPORTED;
    }

    return err;
}

sdc_error_t sdc_intern_formatted_read_header (
    sdc_form_header_generic_t *gen_header,
    const void *formatted_ptr, size_t formatted_len)
{

    sdc_error_t err = SDC_OK;
    const sdc_form_header_ext_t *main_header;
    const sdc_form_header_key_info_common_v0_t *key_common_header;
    const sdc_form_header_key_info_keystore_v0_t *key_keystore_header;
    const sdc_form_header_key_info_builtin_v0_t *key_builtin_header;
    size_t key_info_len;
    sdc_form_operation_t operation;

    size_t remaining;
    size_t offset;

    offset = 0;
    remaining = formatted_len;

    gen_header->header_len = 0;
    gen_header->overall_formatted_len = 0;

    if (remaining < sizeof(sdc_form_header_ext_t)) {
        err = SDC_FORMATTED_DATA_INVALID;
    } else {
        main_header = (const sdc_form_header_ext_t*)(void *)((uint8_t*)formatted_ptr + offset);
        offset += sizeof(sdc_form_header_ext_t);
        remaining -= sizeof(sdc_form_header_ext_t);
        gen_header->header_len += sizeof(sdc_form_header_ext_t);
        gen_header->overall_formatted_len += sizeof(sdc_form_header_ext_t);

        if (ntohl(main_header->magic) != (uint32_t)SDC_FORMATTED_MAGIC)
            err = SDC_FORMATTED_DATA_INVALID;

        if (err == SDC_OK) {
            key_info_len = ntohl(main_header->key_info_len);
            switch (ntohl(main_header->key_info_version)) {
            case 0:
                /* VERSION 0*/
                /* check if this is version 0 key info */
                if ((remaining < sizeof(sdc_form_header_key_info_common_v0_t)) ||
                    (key_info_len < sizeof(sdc_form_header_key_info_common_v0_t))) {
                    err = SDC_FORMATTED_DATA_INVALID;
                } else {
                    key_common_header = (const sdc_form_header_key_info_common_v0_t*)(void *)((uint8_t*)formatted_ptr + offset);
                    offset += sizeof(sdc_form_header_key_info_common_v0_t);
                    key_info_len -= sizeof(sdc_form_header_key_info_common_v0_t);
                    remaining -= sizeof(sdc_form_header_key_info_common_v0_t);
                    gen_header->header_len += sizeof(sdc_form_header_key_info_common_v0_t);
                    gen_header->overall_formatted_len += sizeof(sdc_form_header_key_info_common_v0_t);

                    gen_header->key_info.common.type = (sdc_form_key_type_t)ntohl(key_common_header->key_type);

                    switch (gen_header->key_info.common.type) {
                    case SDC_FORMATTED_KEY_STORAGE:
                        if ((remaining < sizeof(sdc_form_header_key_info_keystore_v0_t)) ||
                            (key_info_len < sizeof(sdc_form_header_key_info_keystore_v0_t))) {
                            err = SDC_FORMATTED_DATA_INVALID;
                        } else {
                            key_keystore_header = (const sdc_form_header_key_info_keystore_v0_t*)(void *)((uint8_t*)formatted_ptr + offset);
                            offset += sizeof(sdc_form_header_key_info_keystore_v0_t);
                            key_info_len -= sizeof(sdc_form_header_key_info_keystore_v0_t);
                            remaining -= sizeof(sdc_form_header_key_info_keystore_v0_t);
                            gen_header->header_len += sizeof(sdc_form_header_key_info_keystore_v0_t);
                            gen_header->overall_formatted_len += sizeof(sdc_form_header_key_info_keystore_v0_t);

                            gen_header->key_info.keystore.kid = (sdc_key_id_t)ntohl(key_keystore_header->kid);
                        }
                        break;
                    case SDC_FORMATTED_KEY_BUILT_IN:
                        if ((remaining < sizeof(sdc_form_header_key_info_builtin_v0_t)) ||
                            (key_info_len < sizeof(sdc_form_header_key_info_builtin_v0_t))) {
                            err = SDC_FORMATTED_DATA_INVALID;
                        } else {
                            key_builtin_header = (const sdc_form_header_key_info_builtin_v0_t*)(void *)((uint8_t*)formatted_ptr + offset);
                            offset += sizeof(sdc_form_header_key_info_builtin_v0_t);
                            key_info_len -= sizeof(sdc_form_header_key_info_builtin_v0_t);
                            remaining -= sizeof(sdc_form_header_key_info_builtin_v0_t);
                            gen_header->header_len += sizeof(sdc_form_header_key_info_builtin_v0_t);
                            gen_header->overall_formatted_len += sizeof(sdc_form_header_key_info_builtin_v0_t);

                            sdc_intern_formatted_form_uid_to_sdc_uid(&err, key_builtin_header->uid, &gen_header->key_info.builtin.perms.uid);
                            sdc_intern_formatted_form_gid_to_sdc_gid(&err, key_builtin_header->gid, &gen_header->key_info.builtin.perms.gid);
                            gen_header->key_info.builtin.key_fmt = (sdc_key_fmt_t)ntohl(key_builtin_header->secret_type);
                            gen_header->key_info.builtin.key_len_bits = ntohl(key_builtin_header->key_len_bits);
                            gen_header->key_info.builtin.key_len_bytes = ntohl(key_builtin_header->key_len_bytes);
                            sdc_intern_formatted_form_perm_to_sdc_perm(&err, key_builtin_header->perms_owner, &gen_header->key_info.builtin.perms.perms_owner);
                            sdc_intern_formatted_form_perm_to_sdc_perm(&err, key_builtin_header->perms_group, &gen_header->key_info.builtin.perms.perms_group);
                            sdc_intern_formatted_form_perm_to_sdc_perm(&err, key_builtin_header->perms_others, &gen_header->key_info.builtin.perms.perms_others);
                            sdc_intern_formatted_form_perm_to_sdc_perm(&err, key_builtin_header->perms_inherit_owner, &gen_header->key_info.builtin.perms.perms_inherit_owner);
                            sdc_intern_formatted_form_perm_to_sdc_perm(&err, key_builtin_header->perms_inherit_group, &gen_header->key_info.builtin.perms.perms_inherit_group);
                            sdc_intern_formatted_form_perm_to_sdc_perm(&err, key_builtin_header->perms_inherit_others, &gen_header->key_info.builtin.perms.perms_inherit_others);
                            gen_header->key_info.builtin.key_modifier.len = ntohl(key_builtin_header->modifier_len);
                            sdc_intern_formatted_form_bool_to_sdc_bool(&err, key_builtin_header->secret_modifier, &gen_header->key_info.builtin.key_modifier.is_secret);
                            gen_header->key_info.builtin.key_modifier.data = NULL;

                            /* in case of modifier */
                            if ((remaining < gen_header->key_info.builtin.key_modifier.len) ||
                                (key_info_len < gen_header->key_info.builtin.key_modifier.len)) {
                                err = SDC_FORMATTED_DATA_INVALID;
                            } else {
                                gen_header->key_info.builtin.key_modifier.data = (uint8_t*)formatted_ptr + offset;
                                offset += gen_header->key_info.builtin.key_modifier.len;
                                key_info_len -= gen_header->key_info.builtin.key_modifier.len;
                                remaining -= gen_header->key_info.builtin.key_modifier.len;
                                gen_header->header_len += gen_header->key_info.builtin.key_modifier.len;
                                gen_header->overall_formatted_len += gen_header->key_info.builtin.key_modifier.len;
                            }
                        }
                        break;
                    default:
                        err = SDC_FORMATTED_DATA_INVALID;
                    }

                    if (key_info_len != 0) {
                        /* some bytes in key header not belonging to anyone */
                        err = SDC_FORMATTED_DATA_INVALID;
                    }
                }
                break;
            default:
                err = SDC_FORMATTED_DATA_INVALID;
            }
        }

        if (err == SDC_OK) {
            operation = (sdc_form_operation_t)ntohl(main_header->format_type);
            if ((operation < SDC_FORMATTED_TYPE_FIRST) || (operation > SDC_FORMATTED_TYPE_LAST)) {
                err = SDC_FORMATTED_DATA_INVALID;
            } else {
                err = sdc_intern_formatted_read_header_opswitch (
                    gen_header, formatted_ptr, main_header,
                    operation, &remaining, &offset);
            }
        }
    }

    if (err == SDC_OK) {
        if (remaining != 0)
            err = SDC_FORMATTED_DATA_INVALID;
    }

    return err;
}

sdc_error_t sdc_intern_check_permission_range(const sdc_permissions_t *perms)
{
    if (perms->perms_owner &
        ~(SDC_PERM_UID_MAX))
        return SDC_PERM_INVALID;
    if (perms->perms_group &
        ~(SDC_PERM_GID_MAX))
        return SDC_PERM_INVALID;
    if (perms->perms_others &
        ~(SDC_PERM_OTHERS_MAX))
        return SDC_PERM_INVALID;
    if (perms->perms_inherit_owner &
        ~(SDC_PERM_INHERIT_UID_MAX))
        return SDC_PERM_INVALID;
    if (perms->perms_inherit_group &
        ~(SDC_PERM_INHERIT_GID_MAX))
        return SDC_PERM_INVALID;
    if (perms->perms_inherit_others &
        ~(SDC_PERM_INHERIT_OTHERS_MAX))
        return SDC_PERM_INVALID;

    return SDC_OK;
}

SDC_INTERN_API sdc_error_t sdc_intern_check_key_enc(sdc_keysecret_enc_t enc, sdc_key_fmt_t fmt)
{
    sdc_error_t err = SDC_OK;

    //TODO : Check if this is to restrictive and should be handle in architecture specific part
    switch(fmt) {
    case SDC_KEY_FMT_SIMPLE_SYM:
        if (enc != SDC_KEY_ENC_PLAIN)
            return SDC_KEY_ENC_INVALID;
        break;
    case SDC_KEY_FMT_RSA_PRIVATE:
    case SDC_KEY_FMT_RSA_PUBLIC:
        if ((enc != SDC_KEY_ENC_DER) && (enc != SDC_KEY_ENC_PEM))
            return SDC_KEY_ENC_INVALID;
        break;
    default:
        err = SDC_KEY_FMT_INVALID;
    }

    return err;
}

sdc_error_t sdc_intern_check_sym_key_fmt(sdc_key_fmt_t fmt)
{
    /* the only symmetric format at the moment */
    if (fmt == SDC_KEY_FMT_SIMPLE_SYM)
        return SDC_OK;

    return SDC_KEY_FMT_INVALID;
}

sdc_error_t sdc_intern_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len)
{
    sdc_error_t err = SDC_OK;
    sdc_form_header_generic_t form_header;
    sdc_key_len_t keylen;

    if (SDC_OK != sdc_intern_check_data_input_buffer(formatted_data, formatted_len))
        err = SDC_FORMATTED_DATA_INVALID;

    if ((secret_mod_data && (secret_mod_len == 0)) ||
        (!secret_mod_data && (secret_mod_len != 0)))
        err = SDC_MODIFIER_INVALID;

    if (!session)
        err = SDC_SESSION_INVALID;

    if (err != SDC_OK)
        return err;

    err = sdc_intern_formatted_read_header (
        &form_header,
        formatted_data, formatted_len);

    if (err == SDC_OK) {
        switch (form_header.key_info.common.type) {
        case SDC_FORMATTED_KEY_STORAGE:
            if (secret_mod_data)
                err = SDC_MODIFIER_INVALID;
            else
                err = sdc_session_load_storage_key(
                    session,
                    form_header.key_info.keystore.kid);
            break;
        case SDC_FORMATTED_KEY_BUILT_IN:
            err = sdc_key_len_from_bits(form_header.key_info.builtin.key_len_bits, &keylen);
            if (err != SDC_OK)
                break;

            if (form_header.key_info.builtin.key_modifier.is_secret) {
                if (!secret_mod_data) {
                    err = SDC_AUTOLOAD_KEY_WITH_SECRET;
                } else {
                    err = sdc_session_load_builtin_key(session,
                                                       form_header.key_info.builtin.key_fmt,
                                                       keylen,
                                                       secret_mod_data, secret_mod_len,
                                                       form_header.key_info.builtin.key_modifier.is_secret,
                                                       &form_header.key_info.builtin.perms);
                }
            } else {
                if (secret_mod_data) {
                    err = SDC_MODIFIER_INVALID;
                } else {
                    err = sdc_session_load_builtin_key(session,
                                                       form_header.key_info.builtin.key_fmt,
                                                       keylen,
                                                       form_header.key_info.builtin.key_modifier.data,
                                                       form_header.key_info.builtin.key_modifier.len,
                                                       form_header.key_info.builtin.key_modifier.is_secret,
                                                       &form_header.key_info.builtin.perms);
                }
            }
            break;
        default:
            err = SDC_AUTOLOAD_KEY_UNSUPPORTED;
        }
    }

    return err;
}

void sdc_intern_overwrite_secret(uint8_t *secret, size_t secretlen)
{
    memset(secret, 0, secretlen);
    asm volatile ("" ::: "memory");
}

sdc_error_t sdc_intern_key_lens_fmt(
    const sdc_key_desc_t *key_desc,
    sdc_key_fmt_t *sup_key_fmt_protect,
    sdc_key_fmt_t *sup_key_fmt_unprotect,
    sdc_key_len_bmsk_t *sup_key_lens,
    sdc_key_len_t *dflt_key_len)
{
    if (sup_key_fmt_protect)
        *sup_key_fmt_protect = key_desc->sup_key_fmt_protect;

    if (sup_key_fmt_unprotect)
        *sup_key_fmt_unprotect = key_desc->sup_key_fmt_unprotect;

    if (sup_key_lens)
        *sup_key_lens = key_desc->sup_key_lens;

    if (dflt_key_len)
        *dflt_key_len = key_desc->dflt_key_len;

    return SDC_OK;
}

sdc_error_t sdc_intern_add_padding_func(
    const sdc_padding_t padding,
    sdc_pad_add_func_t *pad_func)
{
    sdc_error_t err = SDC_OK;

    switch (padding) {
    case SDC_PADDING_INTERNAL:
    case SDC_PADDING_NO:
    case SDC_PADDING_HIDDEN:
        *pad_func = NULL;
        break;
    case SDC_PADDING_PKCS7:
        *pad_func = &sdc_intern_add_pkcs7_padding;
        break;
    default:
        err = SDC_NOT_SUPPORTED;
    }

    return err;
}


sdc_error_t sdc_intern_pad(
    const sdc_padding_t padding,
    uint8_t *buffer, size_t data_len, size_t buffer_len)
{
    sdc_error_t err = SDC_OK;

    switch (padding) {
    case SDC_PADDING_NO:
    case SDC_PADDING_HIDDEN:
        if (data_len != buffer_len) {
            /* something is strange here */
            err = SDC_INTERNAL_ERROR;
        }
        break;
    case SDC_PADDING_PKCS7:
        if (data_len >= buffer_len) {
            /* something is strange here */
            err = SDC_INTERNAL_ERROR;
        } else {
            err = sdc_intern_add_pkcs7_padding(buffer, data_len, buffer_len - data_len);
        }
        break;
    case SDC_PADDING_INTERNAL:
    /* don't use this function in case of SDC_PADDING_INTERNAL */
    /* fall-through */
    default:
        /* this propably means that a new value has been added to sdc_padding_t
         * without handling it here */
        err = SDC_INTERNAL_ERROR;
    }

    return err;
}

sdc_error_t sdc_intern_unpad(
    const sdc_padding_t padding,
    uint8_t *buffer, size_t *data_len, size_t block_len)
{
    sdc_error_t err = SDC_OK;

    switch (padding) {
    case SDC_PADDING_NO:
    case SDC_PADDING_HIDDEN:
        /* nothing to do */
        break;
    case SDC_PADDING_PKCS7:
        err = sdc_intern_remove_pkcs7_padding(buffer, data_len, block_len);
        break;
    case SDC_PADDING_INTERNAL:
    /* don't use this function in case of SDC_PADDING_INTERNAL */
    /* fall-through */
    default:
        /* this propably means that a new value has been added to sdc_padding_t
         * without handling it here */
        err = SDC_INTERNAL_ERROR;
    }

    return err;
}

sdc_error_t sdc_intern_get_cipher_len(
    const sdc_padding_t padding,
    size_t plain_len,
    size_t block_len,
    size_t *cipher_len)
{
    sdc_error_t err = SDC_OK;
    size_t padding_bytes;

    switch (padding) {
    case SDC_PADDING_PKCS7:
        padding_bytes = block_len;
        padding_bytes -= plain_len % block_len;
        *cipher_len = plain_len + padding_bytes;
        break;
    case SDC_PADDING_NO:
    case SDC_PADDING_HIDDEN:
        *cipher_len = plain_len;
        break;
    case SDC_PADDING_INTERNAL:
    /* don't use this function in case of SDC_PADDING_INTERNAL */
    /* fall-through */
    default:
        /* this propably means that a new value has been added to sdc_padding_t
         * without handling it here */
        err = SDC_INTERNAL_ERROR;
    }

    return err;
}

sdc_error_t sdc_intern_get_max_plain_len(
    const sdc_padding_t padding,
    size_t cipher_len,
    size_t block_len,
    size_t *max_plain_len)
{
    sdc_error_t err = SDC_OK;

    (void)block_len;

    switch (padding) {
    case SDC_PADDING_PKCS7:
    case SDC_PADDING_NO:
    case SDC_PADDING_HIDDEN:
        /* same length as cipher - as padding still needs to be removed */
        *max_plain_len = cipher_len;
        break;
    case SDC_PADDING_INTERNAL:
    /* don't use this function in case of SDC_PADDING_INTERNAL */
    /* fall-through */
    default:
        /* this probably means that a new value has been added to sdc_padding_t
         * without handling it here */
        err = SDC_INTERNAL_ERROR;
    }

    return err;
}

sdc_error_t sdc_intern_get_plain_cipher_spec(
    const sdc_padding_t padding,
    size_t arch_min,
    size_t arch_max,
    size_t block_len,
    sdc_plain_cipher_spec_t *plain,
    sdc_plain_cipher_spec_t *cipher)
{
    sdc_error_t err = SDC_OK;
    size_t plain_max = 0;
    size_t cipher_min = 0;
    size_t cipher_max = 0;
    bool plain_aligned = false;
    bool cipher_aligned = false;

    switch (padding) {
    case SDC_PADDING_PKCS7:
        cipher_aligned = true;
        plain_aligned = false;

        /* align to next block length */
        cipher_min = arch_min;
        cipher_min += block_len - cipher_min % block_len;

        /* max cipher length needs to be aligned to block_len */
        cipher_max = arch_max;
        cipher_max -= cipher_max % block_len;

        /* we have at least one byte padding */
        plain_max = cipher_max - 1;

        break;
    case SDC_PADDING_NO:
    case SDC_PADDING_HIDDEN:
        /* no padding, no alignment */
        plain_aligned = false;
        cipher_aligned = false;

        cipher_min = arch_min;
        cipher_max = arch_max;
        plain_max = arch_max;
        break;
    case SDC_PADDING_INTERNAL:
    /* don't use this function in case of SDC_PADDING_INTERNAL */
    /* fall-through */
    default:
        /* this propably means that a new value has been added to sdc_padding_t
         * without handling it here */
        err = SDC_INTERNAL_ERROR;
    }

    if ((err == SDC_OK) &&
        ((cipher_min > cipher_max) ||
         (arch_min > plain_max)))
        err = SDC_INTERNAL_ERROR;

    if (plain) {
        plain->min = arch_min; /* always == arch min */
        plain->max = plain_max;
        plain->aligned = plain_aligned;
    }
    if (cipher) {
        cipher->min = cipher_min;
        cipher->max = cipher_max;
        cipher->aligned = cipher_aligned;
    }

    return err;
}

sdc_error_t sdc_intern_get_update_len(
    const sdc_padding_t padding,
    size_t plain_len,
    size_t block_len,
    size_t *cipher_len)
{
    sdc_error_t err = SDC_OK;
    size_t padding_bytes;

    switch (padding) {
    case SDC_PADDING_PKCS7:
    case SDC_PADDING_NO:
        padding_bytes = block_len;
        padding_bytes -= plain_len % block_len;
        *cipher_len = plain_len + padding_bytes;
        break;
    case SDC_PADDING_INTERNAL:
    /* don't use this function in case of SDC_PADDING_INTERNAL */
    /* fall-through */
    default:
        /* this propably means that a new value has been added to sdc_padding_t
         * without handling it here */
        err = SDC_INTERNAL_ERROR;
    }

    return err;
}

void sdc_intern_clear_session_confidential(sdc_session_t *session)
{
    /* make sure the buffer doesn't contain any confidential data */
    sdc_intern_overwrite_secret(session->unaligned_buffer,
                                SDC_ARCH_MAX_UNALIGNED_LEN);
}

static sdc_error_t check_func_ops_sequence(const sdc_session_t *session, sdc_op_crypto_t crypto_op)
{
    /* same crypto_op and at least one INIT or UPDATE is must before calling UPDATE and FINALIZE operation */
    if((session->sequence.crypto_op != crypto_op) || (session->sequence.op_state != SDC_OP_INITILIZED))
        return SDC_OP_SEQUENCE_INVALID;

    return SDC_OK;
}

sdc_error_t sdc_intern_ops_sequence_ctl_check(const sdc_session_t *session, sdc_op_crypto_t crypto_op, sdc_op_state_t op_state)
{
    sdc_error_t err = SDC_OK;
    switch(op_state)
    {
    case SDC_NO_OP:
        break;
    case SDC_OP_INITILIZED:
        err = check_func_ops_sequence(session, crypto_op);
        break;
    default:
        err = SDC_OP_SEQUENCE_INVALID;
    }
    return err;
}


sdc_error_t sdc_intern_ops_sequence_ctl_set(sdc_session_t *session, sdc_op_crypto_t crypto_op, sdc_op_state_t op_state)
{
    if(session) {
        session->sequence.crypto_op = crypto_op;
        session->sequence.op_state = op_state;

        return SDC_OK;
    }

    return SDC_SESSION_INVALID;
}
